      program Monte_Carlo
      implicit none

c
c     HiPCiP Programming Exercise: Two-Dimensional XY Model
c     Coded by Chris
c     25/02/1999
c               

      
c
c     main
c         

      logical monitor, hot
      integer choice, nsweeps, measuredSweeps, K, L, start, n, fn1, fn2
      integer ergodicity, nthreads, lexMC, srbMC, prbUniMC, prbPUniMC
!$    integer OMP_GET_NUM_THREADS
      parameter (lexMC=1, srbMC=2, prbUniMC=3, prbPUniMC=4, K=100)
      parameter (fn1=10, fn2=12)
      double precision theta(K,K), T, delta, E, sqE, errE, acceptance
      double precision C_v, sweepE, sMeasureAverageE, pMeasureAverageE
      double precision avE, avSqE

c     Seed random number generators
      call rinit(150275)
      write (*,*) 'Two-Dimensional XY Model by Christian Buth'   
      nthreads = 1            ! If compiling without OpenMP
!$OMP PARALLEL DEFAULT(NONE) SHARED(nthreads)
!$OMP SINGLE
!$    nthreads = OMP_GET_NUM_THREADS()
!$    write (*,*) 'Using',nthreads,' threads in ',
!$   &            'PARALLEL regions.'
!$OMP END SINGLE
!$OMP END PARALLEL
      do n=1, nthreads
         call prinit(150275 * n,n-1)
      end do
 1    write (*,*)

c     Menu
      write (*,*) '   1. Lexicographic Monte Carlo simulation'
      write (*,*) '   2. Serial red-black Monte Carlo simulation'
      write (*,*) '   3. Parallel red-black Monte Carlo simulation '
      write (*,*) '      (Using serial random number generator)'     
      write (*,*) '   4. Parallel red-black Monte Carlo simulation '
      write (*,*) '      (Using parallel random number generator)'     
      write (*,*)
      read (*,*) choice      
      if (choice.gt.prbPUniMC.OR.choice.lt.lexMC) then
         write (*,*) 'There are only two options available'   
         write (*,*)
         go to 1
      end if

c     Read parameters  
      write (*,*) 'Would you like to monitor equilibrarion? (T/F)'
      read (*,*) monitor
      write (*,*) 'Enter number of lines'
      read (*,*) L                        
      if (L.lt.1.or.L.gt.K) then
         write (*,*) 'The system size must lie in the range 1 - ',K
         write (*,*)
         go to 1
      end if
      write (*,*) 'Enter number of sweeps'
      read (*,*) nsweeps
      write (*,*) 'Enter temperature'
      read (*,*) T
      write (*,*) 'Enter delta'
      read (*,*) delta
      write (*,*) 'Hot start? (T/F)'
      read (*,*) hot
      if (.not.monitor) then   
         write (*,*) 'Start measuring at'
         read (*,*) start
         if (start.ge.nsweeps) then
            write (*,*) 'Measurements must extend over more than ',
     &                  'two sweeps!'
            go to 1
         end if
      end if
                       
c     Monte Carlo routine
      E = 0
      sqE = 0       
      ergodicity = 0

c     Set up initial configuration   
      if (choice.le.srbMC) then
         call sInitArray(theta,L,hot)
      else
         call pInitArray(theta,L,hot)
      end if
      open(fn1,file='energy.dat')
      if (.not.monitor) open(fn2,file='spheat.dat')
      
c     Do sweeps            
      do n=1, nsweeps                       
         select case (choice)
            case (lexMC)
               call lexMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
            case (srbMC)
               call srbMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
            case (prbUniMC)
               call prbMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
            case (prbPUniMC)
               call pprbMonteCarlo
     &                           (theta,L,T,delta,acceptance,ergodicity)
         end select
c        Measure expectation values of average energy
         if (choice.le.srbMC) then
            sweepE = sMeasureAverageE(theta,L)
         else
            sweepE = pMeasureAverageE(theta,L)
         end if                              
c        Compute specific heat
         if (n.ge.start.and. .not.monitor) then
            E = E + sweepE
            sqE = sqE + sweepE * sweepE
            measuredSweeps = n - start + 1
            avE = E / measuredSweeps
            avSqE = sqE / measuredSweeps 
            errE = avSqE - avE*avE
            C_v = L*L / (T*T) * errE
            write (fn2,*) n, real(C_v)
         end if
c        Print status
         if (monitor) then
            write (*,*) 'Sweep',n,' Energy',real(sweepE),
     &                  ' Acceptance', real(acceptance),
     &                  ' Ergodicity', real(ergodicity) / (L*L)
            ergodicity = 0
         end if
         write (fn1,*) n, real(sweepE)

      end do                                        
      
      if (.not.monitor) close (fn2)
      close(fn1)          
      call showSpins(theta,L)             

      if (.not.monitor) then   
c        Print expectation value of average energy and specific heat
         errE = sqrt(errE / measuredSweeps)
         write (*,*)
         write (*,*) 'Average energy over', measuredSweeps,' sweeps',
     &               real(avE), ' Error', real(errE)
         write (*,*) 'Specific heat ',real(C_v),' Ergodicity',
     &               real(ergodicity)/(L*L*(nsweeps+1))
      end if
      end program Monte_Carlo      


c
c     Data processing procedures
c


c
c     Serially measure average energy per site
c                                    

      double precision function sMeasureAverageE(theta,L)
      implicit none
      integer i, j, L, up, down
      double precision theta(L,L), E
                                     
c     Measure average energy and its square
      E = 0              
      do j=1, L
         do i=1, L  
            E = E - 0.5*(cos(theta(i,j) - theta(up(i,L),j))
     &                 + cos(theta(i,j) - theta(down(i,L),j))
     &                 + cos(theta(i,j) - theta(i,up(j,L)))
     &                 + cos(theta(i,j) - theta(i,down(j,L))))
         end do
      end do                       
      sMeasureAverageE = E / (L*L)
      end function sMeasureAverageE


c
c     Parallelly measure average energy per site
c                                    

      double precision function pMeasureAverageE(theta,L)
      implicit none
      integer i, j, L, up, down
      double precision theta(L,L), E
                                     
c     Measure average energy and its square
      E = 0              
!$OMP PARALLEL DO DEFAULT(NONE) SHARED(theta,L) PRIVATE(i,j)
!$OMP&REDUCTION(+: E)
      do j=1, L
         do i=1, L  
            E = E - 0.5*(cos(theta(i,j) - theta(up(i,L),j))
     &                 + cos(theta(i,j) - theta(down(i,L),j))
     &                 + cos(theta(i,j) - theta(i,up(j,L)))
     &                 + cos(theta(i,j) - theta(i,down(j,L))))
         end do
      end do                   
!$OMP END PARALLEL DO    
      pMeasureAverageE = E / (L*L)
      end function pMeasureAverageE


c
c     Serial initialize array
c
                      
      subroutine sInitArray(theta,L,hot)
      implicit none
      integer i, j, L
      double precision theta(L,L), Pi
      real uni
      logical hot
      Pi = acos(-1.0)                                   
                 
      if (hot) then           
c        Initialize array for hot start
         do j=1, L
            do i=1, L    
               theta(i,j) = 2.0*Pi*(uni() - 0.5)
            end do
         end do
      else
c        Initialize array for cold start
         do j=1, L
            do i=1, L    
               theta(i,j) = 0.0
            end do
         end do
      end if
      end subroutine sInitArray


c
c     Parallel initialize array
c
                      
      subroutine pInitArray(theta,L,hot)
      implicit none
      integer i, j, L, threadno
!$    integer OMP_GET_THREAD_NUM
      double precision theta(L,L), Pi
      real puni
      logical hot
      Pi = acos(-1.0)
      threadno = 0            ! If compiling without OpenMP
            
      if (hot) then           
!$OMP    PARALLEL DEFAULT(NONE) SHARED(theta,Pi,L)
!$OMP&   PRIVATE(i,j,threadno)
!$       threadno = OMP_GET_THREAD_NUM()
c        Initialize array for hot start
!$OMP    DO
         do j=1, L
            do i=1, L    
               theta(i,j) = 2.0*Pi*(puni(threadno) - 0.5)
            end do
         end do  
!$OMP    END DO
!$OMP    END PARALLEL
      else
c        Initialize array for cold start
!$OMP    PARALLEL DO DEFAULT(NONE) SHARED(theta,Pi,L) PRIVATE(i,j)
         do j=1, L
            do i=1, L    
               theta(i,j) = 0.0
            end do
         end do
!$OMP    END PARALLEL DO        
      end if
      end subroutine pInitArray
                 
                 
c
c     Move up in array
c                     

      integer function up(i,L)
      implicit none
      integer i, L
      
      up = mod(i,L) + 1
      end function up
      
      
c
c     Move down in array
c                     

      integer function down(i,L)
      implicit none
      integer i, L
      
      down = mod(i + L - 2,L) + 1
      end function up                   
      
      
c
c     Visualization of spins
c                           

      subroutine showSpins(theta,L)
      implicit none
      integer i, j, L, fn
      double precision theta(L,L)
      parameter (fn=11)
                                  
c     Write one site per line
      open(fn,file='xy2d.dat')
      do j=1, L
         do i=1, L
            write (fn,*) real(cos(theta(i,j))), real(sin(theta(i,j)))
         end do
      end do        
      close(fn)                          
      end subroutine showSpins      
      
      
c
c     Monte Carlo sweep routines
c                   


c
c     One lexicographic Monte Carlo sweep
c
                           
      subroutine lexMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
      implicit none                           
      integer i, j, L, naccepted, nproposed, up, down, ergodicity
      double precision theta(L,L), pi, change, deltaE, E1, E2, delta, T
      double precision acceptance
      real uni
                       
c     Initialize                       
      Pi = acos(-1.0)                                   
      nproposed = 0
      naccepted = 0

c     Process all sites
      do j=1, L
         do i=1, L
c           Propose change and compute energy of old and new configuration
            change = uni() - 0.5
            if (change.ge.0) then
               ergodicity = ergodicity + 1
            else
               ergodicity = ergodicity - 1
            end if
            change = delta*2.0*Pi*change + theta(i,j)
            E1 = - (cos(theta(i,j) - theta(up(i,L),j))
     &           +  cos(theta(i,j) - theta(down(i,L),j))
     &           +  cos(theta(i,j) - theta(i,up(j,L)))
     &           +  cos(theta(i,j) - theta(i,down(j,L))))
            E2 = - (cos(change - theta(up(i,L),j))
     &           +  cos(change - theta(down(i,L),j))
     &           +  cos(change - theta(i,up(j,L)))
     &           +  cos(change - theta(i,down(j,L))))
            deltaE = E2 - E1
c           Accept reject step     
            if (deltaE.le.0.0) then
               theta(i,j) = change
            else
               nproposed = nproposed + 1 
               if (uni().lt.exp(-deltaE/T)) then
                  theta(i,j) = change   
                  naccepted = naccepted + 1
               end if
            end if    
         end do
      end do    
      acceptance = real(naccepted) / nproposed
      end subroutine lexMonteCarlo      


c
c     One serial red-black Monte Carlo sweep
c
                           
      subroutine srbMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
      implicit none                           
      integer i, j, n, L, naccepted, nproposed, up, down, ergodicity
      double precision theta(L,L), pi, change, deltaE, E1, E2, delta, T
      double precision acceptance
      real uni
                       
c     Initialize                       
      Pi = acos(-1.0)                                   
      nproposed = 0
      naccepted = 0

c     Process all sites
      do n=0, 1    
c        Process i+j even sites first.  Then process the i+j odd sites.
         do j=1, L
            do i=1+mod(j+n,2), L, 2
c              Propose change and compute energy of old and new configuration
               change = uni() - 0.5
               if (change.ge.0) then
                  ergodicity = ergodicity + 1
               else
                  ergodicity = ergodicity - 1
               end if
               change = delta*2.0*Pi*change + theta(i,j)
               E1 = - (cos(theta(i,j) - theta(up(i,L),j))
     &              +  cos(theta(i,j) - theta(down(i,L),j))
     &              +  cos(theta(i,j) - theta(i,up(j,L)))
     &              +  cos(theta(i,j) - theta(i,down(j,L))))
               E2 = - (cos(change - theta(up(i,L),j))
     &              +  cos(change - theta(down(i,L),j))
     &              +  cos(change - theta(i,up(j,L)))
     &              +  cos(change - theta(i,down(j,L))))
               deltaE = E2 - E1
c              Accept reject step     
               if (deltaE.le.0.0) then
                  theta(i,j) = change
               else
                  nproposed = nproposed + 1 
                  if (uni().lt.exp(-deltaE/T)) then
                     theta(i,j) = change   
                     naccepted = naccepted + 1
                  end if
               end if    
            end do
         end do    
      end do
      acceptance = real(naccepted) / nproposed
      end subroutine srbMonteCarlo      


c
c     One parallel red-black Monte Carlo sweep using serial uni()
c
                           
      subroutine prbMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
      implicit none                           
      integer i, j, n, K, L, naccepted, nproposed, up, down, ergodicity
      double precision theta(L,L), pi, change, deltaE, E1, E2, delta, T
      double precision acceptance
      parameter (K=100)
      real uni, uniVal(2,K,K)
                       
c     Initialize                       
      Pi = acos(-1.0)                                   
      nproposed = 0
      naccepted = 0                       
      
!$OMP PARALLEL DEFAULT(NONE) SHARED(L,theta,pi,delta,T,uniVal)
!$OMP&PRIVATE(i,j,n,E1,E2,deltaE,change)

c     Generate random numbers        
!$OMP SINGLE
      do j=1, L
         do i=1, L
            uniVal(1,i,j) = uni()
            uniVal(2,i,j) = uni()
         end do
      end do
!$OMP END SINGLE          

c     Process all sites
      do n=0, 1      
c        Process i+j even sites first.  Then process the i+j odd sites.
!$OMP    DO REDUCTION(+: naccepted,nproposed,ergodicity)
         do j=1, L
            do i=1+mod(j+n,2), L, 2
c              Propose change and compute energy of old and new configuration
               change = uniVal(1,i,j) - 0.5
               if (change.ge.0) then
                  ergodicity = ergodicity + 1
               else
                  ergodicity = ergodicity - 1
               end if
               change = delta*2.0*Pi*change + theta(i,j)
               E1 = - (cos(theta(i,j) - theta(up(i,L),j))
     &              +  cos(theta(i,j) - theta(down(i,L),j))
     &              +  cos(theta(i,j) - theta(i,up(j,L)))
     &              +  cos(theta(i,j) - theta(i,down(j,L))))
               E2 = - (cos(change - theta(up(i,L),j))
     &              +  cos(change - theta(down(i,L),j))
     &              +  cos(change - theta(i,up(j,L)))
     &              +  cos(change - theta(i,down(j,L))))
               deltaE = E2 - E1
c              Accept reject step     
               if (deltaE.le.0.0) then
                  theta(i,j) = change
               else
                  nproposed = nproposed + 1 
                  if (uniVal(2,i,j).lt.exp(-deltaE/T)) then
                     theta(i,j) = change   
                     naccepted = naccepted + 1
                  end if      
               end if    
            end do                
         end do                                
!$OMP    END DO
      end do
!$OMP END PARALLEL
      acceptance = real(naccepted) / nproposed
      end function prbMonteCarlo      


c
c     One parallel red-black Monte Carlo sweep using parallel puni()
c
                           
      subroutine pprbMonteCarlo(theta,L,T,delta,acceptance,ergodicity)
      implicit none                           
      integer i, j, n, K, L, naccepted, nproposed, up, down, ergodicity
      integer threadno
!$    integer OMP_GET_THREAD_NUM
      double precision theta(L,L), pi, change, deltaE, E1, E2, delta, T
      double precision acceptance
      parameter (K=100)
      real puni
                       
c     Initialize                       
      Pi = acos(-1.0)                                   
      nproposed = 0
      naccepted = 0
      threadno = 0            ! If compiling without OpenMP                 
      
c     Process all sites
      do n=0, 1      
!$OMP    PARALLEL DEFAULT(NONE) SHARED(n,L,theta,pi,delta,T)
!$OMP&   PRIVATE(i,j,E1,E2,deltaE,change,threadno)
!$       threadno = OMP_GET_THREAD_NUM()
c        Process i+j even sites first.  Then process the i+j odd sites.
!$OMP    DO REDUCTION(+: naccepted,nproposed,ergodicity)
         do j=1, L
            do i=1+mod(j+n,2), L, 2
c              Propose change and compute energy of old and new configuration
               change = puni(threadno) - 0.5
               if (change.ge.0) then
                  ergodicity = ergodicity + 1
               else
                  ergodicity = ergodicity - 1
               end if
               change = delta*2.0*Pi*change + theta(i,j)
               E1 = - (cos(theta(i,j) - theta(up(i,L),j))
     &              +  cos(theta(i,j) - theta(down(i,L),j))
     &              +  cos(theta(i,j) - theta(i,up(j,L)))
     &              +  cos(theta(i,j) - theta(i,down(j,L))))
               E2 = - (cos(change - theta(up(i,L),j))
     &              +  cos(change - theta(down(i,L),j))
     &              +  cos(change - theta(i,up(j,L)))
     &              +  cos(change - theta(i,down(j,L))))
               deltaE = E2 - E1
c              Accept reject step     
               if (deltaE.le.0.0) then
                  theta(i,j) = change
               else
                  nproposed = nproposed + 1 
                  if (puni(threadno).lt.exp(-deltaE/T)) then
                     theta(i,j) = change   
                     naccepted = naccepted + 1
                  end if      
               end if    
            end do
         end do      
!$OMP    END DO                          
!$OMP    END PARALLEL
      end do
      acceptance = real(naccepted) / nproposed
      end subroutine pprbMonteCarlo
