      program Fluid_Dynamics
      implicit none

c     Title: Practical: Fluid Dynamics 
c     Author: Christian Buth
c     Date: 23/02/1999


c
c     Constants
c

      double precision epsilon
      integer i, j, b0, w0, h0, K, L0, fn1, fn2
      integer sJacI, slGSI, sloGSI, srbGSI, pJacI, prbGSI
      integer srbGSV, srboGSV, prbGSV, prboGSV
      parameter (epsilon=1E-6, K=100, b0=5, h0=15, w0=5)
      parameter (L0=30, fn1=10, fn2=11)
      parameter (sJacI=1, slGSI=2, sloGSI=3, srbGSI=4, pJacI=5)
      parameter (prbGSI=6, srbGSV=7, srboGSV=8, prbGSV=9, prboGSV=10)

c
c     Main
c

      integer n, b, w, h, L, choice
!$    integer OMP_GET_NUM_THREADS
      double precision Psi(K,K), Zeta(K,K)
      double precision resPsi, resZeta, omega, Re, alpha
      
c     Initialisation
      write (*,*) 'Fluid Dynamics by Christian Buth'
!$OMP PARALLEL DEFAULT(NONE)
!$OMP SINGLE
!$    write (*,*) 'Using',OMP_GET_NUM_THREADS(),' threads in ',
!$   &            'PARALLEL regions.'
!$OMP END SINGLE
!$OMP END PARALLEL
 1    write (*,*)

c     Menu
      write (*,*) 'Choose a PDE solver for inviscid flow:'
      write (*,*) '   1. Serial Jacobi'
      write (*,*) '   2. Serial lexicographic Gauss-Seidel'
      write (*,*) '   3. Serial lexicographic overrelaxed Gauss-Seidel'
      write (*,*) '   4. Serial red-black Gauss-Seidel'
      write (*,*) '   5. Parallel Jacobi'
      write (*,*) '   6. Parallel red-black Gauss-Seidel'
      write (*,*)
      write (*,*) 'Choose a PDE solver for viscous flow:'
      write (*,*) '   7. Serial red-black Gauss-Seidel'
      write (*,*) '   8. Serial overrelaxed red-black Gauss-Seidel'
      write (*,*) '   9. Parallel red-black Gauss-Seidel'
      write (*,*) '  10. Parallel overrelaxed red-black Gauss-Seidel'
      write (*,*)
      read (*,*) choice
      write (*,*)
      if (choice.gt.prboGSV.OR.choice.lt.sJacI) then
         write (*,*) 'There are only ',prboGSV,' PDE solver available!'
         go to 1
      end if

c     Specify system size and scale variables
      write (*,*) 'System size (maximum',K,'): '
      read (*,*) L
      write (*,*)
      if (L.lt.10.or.L.gt.K) then
         write (*,*) 'Array size 10 till ', K
         go to 1
      end if
      alpha = real(L) / L0
      b = alpha * b0
      h = alpha * h0
      w = alpha * w0

c     Ask for additional parameters
      if(choice.eq.sloGSI.or.choice.eq.srboGSV.or.choice.eq.prboGSV)then
         write (*,*) 'Specify overelaxation parameter:'
         read (*,*) omega
         write (*,*)
      end if
      if (choice.ge.srbGSV) then
         write (*,*) 'Specify Reynold''s number:'
         read (*,*) Re
         write (*,*)
         Re = Re / alpha                                       ! Scale correctly
      end if

c     Initialize stream function array
      if (choice.le.srbGSI.or.choice.eq.srbGSV.or.choice.eq.srboGSV)then 
         call sbcStreamFunction(Psi,b,w,h,L)
      else
         call pbcStreamFunction(Psi,b,w,h,L)
      end if       
c     Initialize vorticity array
      if (choice.eq.srbGSV.or.choice.eq.srboGSV) then 
         do j=1, L
            do i=1, L
               Zeta(i,j) = 0
            end do
         end do
      else if (choice.eq.prbGSV.or.choice.eq.prboGSV) then
!$OMP PARALLEL DO PRIVATE (i,j) SHARED(Zeta,L) DEFAULT(NONE)
         do j=1, L
            do i=1, L
               Zeta(i,j) = 0
            end do
         end do
!$OMP END PARALLEL DO
      end if
                                       

c
c     Solution of equations
c

c     Graph: residue of stream function against number of iterations
      open(fn1,file='psiRes.dat')
c     Graph: residue of vorticity against number of iterations
      if (choice.ge.srbGSV) open(fn2,file='zetaRes.dat')
      resPsi  = 1                                             ! To start looping
      resZeta = 0
      n = 0

c     Solve partial differential equations
      do while (resPsi.gt.epsilon.or.resZeta.gt.epsilon)
c        Choose partial differential equation solver
         select case (choice)
            case (sJacI)
               call sJacobiI(Psi,L)
            case (slGSI)
               call lGaussI(Psi,L)
            case (sloGSI)
               call loGaussI(Psi,omega,L)
            case (srbGSI)
               call srbGaussI(Psi,L)
            case (pJacI)
               call pJacobiI(Psi,L)
            case (prbGSI)
               call prbGaussI(Psi,L)
            case (srbGSV)
               call srbGaussV(Psi,Zeta,Re,L)
            case (srboGSV)
               call srboGaussV(Psi,Zeta,omega,Re,L)
            case (prbGSV)
               call prbGaussV(Psi,Zeta,Re,L)
            case (prboGSV)
               call prboGaussV(Psi,Zeta,omega,Re,L)
         end select
c        Calculate and write out residues
         select case (choice)
            case (sJacI:srbGSI)
               call sResidueI(Psi,resPsi,L)
            case (pJacI,prbGSI)
               call pResidueI(Psi,resPsi,L)
            case (srbGSV,srboGSV)
               call sResidueV(Psi,Zeta,Re,resPsi,resZeta,L)
            case (prbGSV,prboGSV)
               call pResidueV(Psi,Zeta,Re,resPsi,resZeta,L)
         end select
         n = n + 1
         write (fn1,*) n, real(log10(resPsi))
         if (choice.ge.srbGSV) write (fn2,*) n, real(log10(resZeta))
      end do
      if (choice.ge.srbGSV) close(fn2)
      close(fn1)

c     Visualisation
      if (choice.le.srbGSI.or.choice.eq.srbGSV.or.choice.eq.srboGSV)then 
         call sCalcVel(Psi,b,h,w,L)
      else
         call pCalcVel(Psi,b,h,w,L)
      end if

c     Display accuracy
      write (*,*) 'The simulation needed', n, ' iterations to push'
      write (*,*) 'accuracy to', real(resPsi), ' for psi.'
      if (choice.ge.srbGSV) write (*,*) 'Accuracy of zeta is ',
     &                             real(resZeta),'.'
      end program Fluid_Dynamics


c
c     Data processing procedures
c


c
c     Serial impose boundary conditions for stream function
c

      subroutine sbcStreamFunction(Psi,b,w,h,L)
      implicit none
      integer i, j, b, w, h, L
      double precision Psi(L,L)

      do j=1, L
         do i=1, L
            Psi(i,j) = 0
         end do
      end do
      do i=b+1, b+w-1
         Psi(i,1) = i - b
      end do
      do i=b+w, L
         Psi(i,1) = w
      end do
      do j=1, h
         Psi(L,j) = w
      end do
      do i=h+1, h+w-1
         Psi(L,j) = w-j+h
      end do
      end subroutine sbcStreamFunction


c
c     Parallel impose boundary conditions for stream function
c

      subroutine pbcStreamFunction(Psi,b,w,h,L)
      implicit none
      integer i, j, b, w, h, L
      double precision Psi(L,L)

!$OMP PARALLEL PRIVATE(i,j) SHARED(Psi,b,w,h,L) DEFAULT(NONE)
!$OMP DO
      do j=1, L
         do i=1, L
            Psi(i,j) = 0
         end do
      end do
!$OMP END DO
!$OMP DO
      do i=b+1, b+w-1
         Psi(i,1) = i - b
      end do
!$OMP END DO NOWAIT
!$OMP DO
      do i=b+w, L
         Psi(i,1) = w
      end do
!$OMP END DO NOWAIT
!$OMP DO
      do j=1, h
         Psi(L,j) = w
      end do
!$OMP END DO NOWAIT
!$OMP DO
      do i=h+1, h+w-1
         Psi(L,j) = w-j+h
      end do
!$OMP END DO
!$OMP END PARALLEL
      end subroutine pbcStreamFunction


c
c     Serial computation of velocity field
c

      subroutine sCalcVel(Psi,b,h,w,L)
      implicit none
      integer i, j, b, h, w, L, fn, K
      parameter (K=100, fn=12)
      double precision Psi(L,L), ux(K,K), uy(K,K)

c     Set boundary velocity
      do j=1, L
         do i=1, L
            ux(i,j) = 0
            uy(i,j) = 0
         end do
      end do
      do i=b+1, b+w-1
         uy(i,1) = -1
      end do
      do j=h+1, h+w-1
         ux(1,j) = -1
      end do

c     Calculate velocities
      do j=2, L-1
         do i=2, L-1
            ux(i,j) = 0.5 * (Psi(i,j+1) - Psi(i,j-1))
            uy(i,j) = 0.5 * (Psi(i-1,j) - Psi(i+1,j))
         end do
      end do

c     Write velocities
      open(fn,file='fluid.dat')
      do j=1, L
         do i=1, L
            write (fn,*) real(ux(i,j)),real(uy(i,j))
         end do
      end do
      close(fn)
      end subroutine sCalcVel


c
c     Parallel computation of velocity field
c

      subroutine pCalcVel(Psi,b,h,w,L)
      implicit none
      integer i, j, b, h, w, L, fn, K
      parameter (K=100, fn=13)
      double precision Psi(L,L), ux(K,K), uy(K,K)

c     Set boundary velocity
!$OMP PARALLEL PRIVATE(i,j) SHARED(ux,uy,Psi,b,h,w,L) DEFAULT(NONE)
!$OMP DO
      do j=1, L
         do i=1, L
            ux(i,j) = 0
            uy(i,j) = 0
         end do
      end do
!$OMP END DO
!$OMP DO
      do i=b+1, b+w-1
         uy(i,1) = -1
      end do
!$OMP END DO NOWAIT
!$OMP DO
      do j=h+1, h+w-1
         ux(1,j) = -1
      end do
!$OMP END DO

c     Calculate velocities
!$OMP DO
      do j=2, L-1
         do i=2, L-1
            ux(i,j) = 0.5 * (Psi(i,j+1) - Psi(i,j-1))
            uy(i,j) = 0.5 * (Psi(i-1,j) - Psi(i+1,j))
         end do
      end do
!$OMP END DO
!$OMP END PARALLEL

c     Write velocities
      open(fn,file='fluid.dat')
      do j=1, L
         do i=1, L
            write (fn,*) real(ux(i,j)),real(uy(i,j))
         end do
      end do
      close(fn)
      end subroutine pCalcVel


c
c     Serial calculation of residue for inviscid flow
c

      subroutine sResidueI(Psi,res,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L), res

      res = 0
      do j=2, L-1
         do i=2, L-1
            res = res + (Psi(i-1,j) + Psi(i+1,j) +  Psi(i,j-1)
     &                   + Psi(i,j+1) - 4 * Psi(i,j))**2
         end do
      end do
      
      res = sqrt(res)
      end subroutine sResidueI


c
c     Parallel calculation of residue for inviscid flow
c

      subroutine pResidueI(Psi,res,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L), res

      res = 0
!$OMP PARALLEL DO PRIVATE(i,j) SHARED(Psi,L) DEFAULT(NONE)
!$OMP&REDUCTION(+:res)
      do j=2, L-1
         do i=2, L-1
            res = res + (Psi(i-1,j) + Psi(i+1,j) +  Psi(i,j-1)
     &                   + Psi(i,j+1) - 4 * Psi(i,j))**2
         end do
      end do
!$OMP END PARALLEL DO
      
      res = sqrt(res)
      end subroutine pResidueI


c
c     Serial calculation of residue for viscous flow
c

      subroutine sResidueV(Psi,Zeta,Re,resPsi,resZeta,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L), Zeta(L,L), resPsi, resZeta, Re

      resPsi  = 0
      resZeta = 0
      do j=2, L-1
         do i=2, L-1
            resPsi = resPsi + (Psi(i-1,j) + Psi(i+1,j) + Psi(i,j-1)
     &                  + Psi(i,j+1) - Zeta(i,j) - 4.0 * Psi(i,j))**2
            resZeta= resZeta + (Zeta(i-1,j) + Zeta(i+1,j) + Zeta(i,j-1)
     &                + Zeta(i,j+1) - Re/4.0 * ((Psi(i,j+1)
     &                - Psi(i,j-1)) * (Zeta(i+1,j) - Zeta(i-1,j))
     &                - (Psi(i+1,j) - Psi(i-1,j)) * (Zeta(i,j+1)
     &                - Zeta(i,j-1))) - 4.0 * Zeta(i,j))**2
         end do
      end do
      
      resPsi  = sqrt(resPsi)
      resZeta = sqrt(resZeta)
      end subroutine sResidueV


c
c     Parallel calculation of residue for viscous flow
c

      subroutine pResidueV(Psi,Zeta,Re,resPsi,resZeta,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L), Zeta(L,L), resPsi, resZeta, Re

      resPsi  = 0
      resZeta = 0
!$OMP PARALLEL DO PRIVATE (i,j) SHARED(Psi,Zeta,Re,L) DEFAULT(NONE)
!$OMP&REDUCTION(+:resPsi) REDUCTION(+:resZeta)
      do j=2, L-1
         do i=2, L-1
            resPsi = resPsi + (Psi(i-1,j) + Psi(i+1,j) + Psi(i,j-1)
     &                  + Psi(i,j+1) - Zeta(i,j) - 4.0 * Psi(i,j))**2
            resZeta= resZeta + (Zeta(i-1,j) + Zeta(i+1,j) + Zeta(i,j-1)
     &                + Zeta(i,j+1) - Re/4.0 * ((Psi(i,j+1)
     &                - Psi(i,j-1)) * (Zeta(i+1,j) - Zeta(i-1,j))
     &                - (Psi(i+1,j) - Psi(i-1,j)) * (Zeta(i,j+1)
     &                - Zeta(i,j-1))) - 4.0 * Zeta(i,j))**2
         end do
      end do
!$OMP END PARALLEL DO
      
      resPsi  = sqrt(resPsi)
      resZeta = sqrt(resZeta)
      end subroutine pResidueV


c
c     Partial differential equation solver for inviscid flow
c


c
c     Serial Jacobi PDE solver for inviscid flow
c

      subroutine sJacobiI(Psi,L)
      implicit none
      integer i, j, L, K
      parameter (K=100)
      double precision Psi(L,L), Zeta(K,K)

c     Calculate one iteration
      do j=2, L-1
         do i=2, L-1
            Zeta(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                       +  Psi(i,j-1) + Psi(i,j+1))
         end do
      end do

      do i=2, L-1
         do j=2, L-1
            Psi(i,j) = Zeta(i,j)
         end do
      end do
      end subroutine sJacobiI


c
c     Parallel Jacobi PDE solver for inviscid flow
c

      subroutine pJacobiI(Psi,L)
      implicit none
      integer i, j, L, K
      parameter (K=100)
      double precision Psi(L,L), Zeta(K,K)

c     Calculate one iteration

!$OMP PARALLEL PRIVATE(i,j) SHARED(Zeta,Psi,L) DEFAULT(NONE)
!$OMP DO
      do j=2, L-1
         do i=2, L-1
            Zeta(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                       +  Psi(i,j-1) + Psi(i,j+1))
         end do
      end do
!$OMP END DO

!$OMP DO
      do i=2, L-1
         do j=2, L-1
            Psi(i,j) = Zeta(i,j)
         end do
      end do
!$OMP END DO
!$OMP END PARALLEL
      end subroutine pJacobiI


c
c     Lexicographic Gauss-Seidel PDE solver for inviscid flow
c

      subroutine lGaussI(Psi,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L)

c     Calculate one iteration
      do j=2, L-1
         do i=2, L-1
            Psi(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                       + Psi(i,j-1) + Psi(i,j+1))
         end do
      end do
      end subroutine lGaussI


c
c     Lexicographic overrelaxed Gauss-Seidel PDE solver for inviscid flow
c

      subroutine loGaussI(Psi,omega,L)
      implicit none
      integer i, j, L
      double precision Psi(L,L), omega, oneMinusOmega

c     Calculate one iteration
      oneMinusOmega = 1.0 - omega
      do j=2, L-1
         do i=2, L-1
            Psi(i,j) = oneMinusOmega * Psi(i,j) + 0.25 * omega
     &                  * (Psi(i-1,j) + Psi(i+1,j) + Psi(i,j-1)
     &                  + Psi(i,j+1))
         end do
      end do
      end subroutine loGaussI


c
c     Serial red-black Gauss-Seidel PDE solver for inviscid flow
c

      subroutine srbGaussI(Psi,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L)
                      
c     Calculate even iteration then odd iteration
      do k=0, 1
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                          + Psi(i,j-1) + Psi(i,j+1))
            end do
         end do
      end do
      end subroutine srbGaussI


c
c     Parallel red-black Gauss-Seidel PDE solver for inviscid flow
c

      subroutine prbGaussI(Psi,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L)

c     Calculate even iteration then odd iteration
      do k=0, 1
!$OMP    PARALLEL DO PRIVATE(i,j) SHARED(Psi,k,L) DEFAULT(NONE)
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                          + Psi(i,j-1) + Psi(i,j+1))
            end do
         end do
!$OMP    END PARALLEL DO
      end do
      end subroutine prbGaussI


c
c     Partial differential equation solver for viscous flow
c


c
c     Serial red-black Gauss-Seidel PDE solver for viscous flow
c

      subroutine srbGaussV(Psi,Zeta,Re,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L), Zeta(L,L), Re

c     Impose boundary conditions for vorticity
      do i=1, L
         Zeta(i,1) = 2.0 * (Psi(i,2)   - Psi(i,1))
         Zeta(i,L) = 2.0 * (Psi(i,L-1) - Psi(i,L))
         Zeta(1,i) = 2.0 * (Psi(2,i)   - Psi(1,i))
         Zeta(L,i) = 2.0 * (Psi(L-1,i) - Psi(L,i))
      end do

c     Calculate even iteration then odd iteration
      do k=0, 1
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                         + Psi(i,j-1) + Psi(i,j+1) - Zeta(i,j))
               Zeta(i,j)= 0.25 * (Zeta(i-1,j) + Zeta(i+1,j)
     &                         + Zeta(i,j-1) + Zeta(i,j+1))
     &                         - Re/16.0 * ((Psi(i,j+1) - Psi(i,j-1))
     &                         * (Zeta(i+1,j) - Zeta(i-1,j))
     &                         - (Psi(i+1,j) - Psi(i-1,j))
     &                         * (Zeta(i,j+1) - Zeta(i,j-1)))
            end do
         end do
      end do
      end subroutine srbGaussV


c
c     Serial overrelaxed red-black Gauss-Seidel PDE solver for viscous flow
c

      subroutine srboGaussV(Psi,Zeta,omega,Re,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L), Zeta(L,L), Re, omega, oneMinusOmega
      oneMinusOmega = 1.0 - omega

c     Impose boundary conditions for vorticity
      do i=1, L
         Zeta(i,1) = 2.0 * (Psi(i,2)   - Psi(i,1))
         Zeta(i,L) = 2.0 * (Psi(i,L-1) - Psi(i,L))
         Zeta(1,i) = 2.0 * (Psi(2,i)   - Psi(1,i))
         Zeta(L,i) = 2.0 * (Psi(L-1,i) - Psi(L,i))
      end do

c     Calculate even iteration then odd iteration
      do k=0, 1
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = oneMinusOmega * Psi(i,j) + 0.25 * omega
     &                     * (Psi(i-1,j) + Psi(i+1,j) + Psi(i,j-1)
     &                     + Psi(i,j+1) - Zeta(i,j))
               Zeta(i,j)= oneMinusOmega * Zeta(i,j) + 0.25 * omega
     &                     * (Zeta(i-1,j) + Zeta(i+1,j) + Zeta(i,j-1)
     &                     + Zeta(i,j+1))
     &                     - omega*Re/16.0 * ((Psi(i,j+1) - Psi(i,j-1))
     &                     * (Zeta(i+1,j) - Zeta(i-1,j)) - (Psi(i+1,j)
     &                     - Psi(i-1,j)) * (Zeta(i,j+1) - Zeta(i,j-1)))
            end do
         end do
      end do
      end subroutine srboGaussV


c
c     Parallel red-black Gauss-Seidel PDE solver for viscous flow
c

      subroutine prbGaussV(Psi,Zeta,Re,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L), Zeta(L,L), Re

c     Impose boundary conditions for vorticity
!$OMP PARALLEL DO PRIVATE(i) SHARED(Psi,Zeta,L) DEFAULT(NONE)
      do i=1, L
         Zeta(i,1) = 2.0 * (Psi(i,2)   - Psi(i,1))
         Zeta(i,L) = 2.0 * (Psi(i,L-1) - Psi(i,L))
         Zeta(1,i) = 2.0 * (Psi(2,i)   - Psi(1,i))
         Zeta(L,i) = 2.0 * (Psi(L-1,i) - Psi(L,i))
      end do
!$OMP END PARALLEL DO

c     Calculate even iteration then odd iteration
      do k=0, 1
!$OMP    PARALLEL DO PRIVATE(i,j) SHARED(Psi,Zeta,Re,k,L) DEFAULT(NONE)
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = 0.25 * (Psi(i-1,j) + Psi(i+1,j)
     &                         + Psi(i,j-1) + Psi(i,j+1) - Zeta(i,j))
               Zeta(i,j)= 0.25 * (Zeta(i-1,j) + Zeta(i+1,j)
     &                         + Zeta(i,j-1) + Zeta(i,j+1))
     &                         - Re/16.0 * ((Psi(i,j+1) - Psi(i,j-1))
     &                         * (Zeta(i+1,j) - Zeta(i-1,j))
     &                         - (Psi(i+1,j) - Psi(i-1,j))
     &                         * (Zeta(i,j+1) - Zeta(i,j-1)))
            end do
         end do
!$OMP    END PARALLEL DO
      end do
      end subroutine prbGaussV


c
c     Parallel overrelaxed red-black Gauss-Seidel PDE solver for viscous flow
c

      subroutine prboGaussV(Psi,Zeta,omega,Re,L)
      implicit none
      integer i, j, k, L
      double precision Psi(L,L), Zeta(L,L), Re, omega, oneMinusOmega
      oneMinusOmega = 1.0 - omega

c     Impose boundary conditions for vorticity
!$OMP PARALLEL DO PRIVATE(i) SHARED(Psi,Zeta,L) DEFAULT(NONE)
      do i=1, L
         Zeta(i,1) = 2.0 * (Psi(i,2)   - Psi(i,1))
         Zeta(i,L) = 2.0 * (Psi(i,L-1) - Psi(i,L))
         Zeta(1,i) = 2.0 * (Psi(2,i)   - Psi(1,i))
         Zeta(L,i) = 2.0 * (Psi(L-1,i) - Psi(L,i))
      end do
!$OMP END PARALLEL DO

c     Calculate even iteration then odd iteration
      do k=0, 1
!$OMP    PARALLEL DO DEFAULT(NONE) PRIVATE(i,j)
!$OMP&   SHARED(Psi,Zeta,Re,k,L,omega,oneMinusOmega)
         do j=2, L-1
            do i=2+mod(j+k,2), L-1, 2
               Psi(i,j) = oneMinusOmega * Psi(i,j) + 0.25 * omega
     &                     * (Psi(i-1,j) + Psi(i+1,j) + Psi(i,j-1)
     &                     + Psi(i,j+1) - Zeta(i,j))
               Zeta(i,j)= oneMinusOmega * Zeta(i,j) + 0.25 * omega
     &                     * (Zeta(i-1,j) + Zeta(i+1,j) + Zeta(i,j-1)
     &                     + Zeta(i,j+1))
     &                     - omega*Re/16.0 * ((Psi(i,j+1) - Psi(i,j-1))
     &                     * (Zeta(i+1,j) - Zeta(i-1,j)) - (Psi(i+1,j)
     &                     - Psi(i-1,j)) * (Zeta(i,j+1) - Zeta(i,j-1)))
            end do
         end do
!$OMP    END PARALLEL DO
      end do
      end subroutine prboGaussV      
