// TOF.cpp, Version 1.0 vom 4.2.2000
//
// Programm zum Fitten der Time-Of-Flight-Spektren, die im Versuch
// "F17 Experimente zur Neutronenphysik" des Fortgeschrittenen Praktikums an
// der Universitt Heidelberg mit dem Programm "TOF" gemeen wurden.
//
// Berechnung von chi^2
// Christian Buth
// 11.07.2000
//
// Version 1.00: 16.07.2000 (chi^2 Berechnen und Abspeichern, Fehler aus dem
//              vorgegebenen Programm entfernt, theoretische Spektren speichern)
// Version 1.10: 27.10.2000 (Neue Formatierung)
// Version 1.20: 28.10.2000 (Fehler in Lngenangaben fr Dateinamen und Pfaden
//              behoben)


	// Einbinden der notwendigen C-Standard-Bibliotheken
#include <stdio.h>   		// Bibliothek fr Standard Ein-und Ausgaben
#include <stdlib.h>		// Bibliothek fr Speicherverwaltungsfunktionen
#include <string.h>		// Bibliothek fr Stringmanipulation
#include <math.h>    		// Bibliothek fr mathematematische Funktionen


	// Definition der globalen Konstanten
#define k 1.380658e-23          // Boltzmannfaktor
#define m 1.6749286e-27         // Ruhemasse des Neutrons
#define Pi 3.1415927            // Mathematische Konstante Pi
#define Dateinamenlaenge 8      // Lnge von Dateinamen
#define Pfadlaenge 32           // Lnge von Pfadangaben
#define Pfadnamenlaenge 45      // Lnge von Pfadangaben mit Dateinamen
#define TOF_Endung ".asc"       // Endung der vom TOF Programm erzeugten Dateien
#define Ausgabe_Endung ".dat"   // Endung aller ausgegebenen Dateien


        // Makros definieren
#define sqr(x) ((x)*(x))        // x^2 schnell berechnen
#define make_str(s) #s          // s in "s" umwandeln
#define make_str_eval(s) make_str(s) // s auswerten, dann in "s" umwandeln


	// Definition der globalen Variablen und Listen
char dir[Pfadnamenlaenge]      	// Standardverzeichnis der gemeenen Spektren
     ="D:\\FP\\tof\\";

float *messdaten = 0;		// Zeiger auf die Liste der gemeenen Daten
float *fehler = 0;              // Zeiger auf die Liste der zugehrigen Fehler
float *spektrum_theo = 0;       // Zeiger auf die Liste des theoretisch
                                // berechneten Spektrums
float *chi_quadrat = 0;       	// Zeiger auf die Liste der berechneten
                                // Chi-Quadrate

int anzahl_kanaele=0;		// Anzahl der gemeenen Daten
int anzahl_daten=0;             // Anzahl der Kanle mit Medaten, ohne die
                                // Kanlle, in denen nur Untergrund ist
int anzahl_schritte=200;     	// Anzahl der Punkte fr die chi^2 Parabel
int Kanal0=62;                  // Kanalnummer, ab dem der Zerhacker vllig
                                // geffnet ist

float dwell=2.5e-6;             // eingestellte dwelltime [s] der TOF-Meung
float flugstrecke=0.3;          // gemeene Flugstrecke der Neutronen [m]
float frequenz=200.;            // Frequenz der Chopperscheibe [Hz]
float temperatur=300.;          // Temperatur des Neutronenspektrums [K]
float untergrund=1.;		// Untergrund der Meung
float fehler_untergrund=0.1;	// Fehler des Untergrundes
float T_start=250;              // Starttemperatur fr den chi^2 Fit
float Delta_T=5;                // Temperaturschrittweite bei chi^2 Fit


// Routine zum Laden des gemeenen TOF-Spektrums von der Festplatte.
// Neben der Anzahl der benutzten Kanle wird der entsprechende Datei-
// name abgefragt.  Existiert die Datei nicht, so wird eine Fehlermeldung
// erzeugt.  Die Liste mit den gemeenen Daten und die mit den dazugehrigen
// statistischen Fehlern werden als globale Zeiger erzeugt, so dass sie im
// gesamten Programm bekannt sind und auf sie zugegriffen werden kann.

void Laden(void)
  {
    int i;
    FILE *Dateizeiger;
    float wert;
    char name[Dateinamenlaenge+1], file_name[Pfadnamenlaenge+1];

    free(messdaten);		// Freigeben des Speicherplatzes der bisher
    free(fehler);               // benutzten Listen

    printf("Geben Sie bitte die Anzahl der TOF-Kanle an: ");
    scanf("%d",&anzahl_kanaele);

    // Allokieren des bentigten Speicherplatzes
    // medaten ist der Pointer auf die Liste der gemeenen Daten.
    // Fehler ist der Pointer auf die Liste der dazugehrenden statistischen
    // Fehler.
    messdaten = (float *) malloc ((anzahl_kanaele+1)*sizeof(float));
    fehler = (float *) malloc ((anzahl_kanaele+1)*sizeof(float));

    printf("Geben Sie bitte den Namen des TOF-Files an (maximal "
           make_str_eval(Dateinamenlaenge) " Zeichen): ");
    scanf("%" make_str_eval(Dateinamenlaenge) "s",name);

    // Erzeugung des vollstndigen Pfadnamens
    strcpy(file_name, dir);
    strcat(file_name, name);
    strcat(file_name,TOF_Endung);

    // ffnen der Datei und Ueberprfung, ob existent
    Dateizeiger=fopen(file_name,"rt");
    if (Dateizeiger == NULL)
      {
	printf("\nDatei existiert nicht!\n");
	anzahl_kanaele = 0;
	fclose(Dateizeiger);
	return;
      }

    // Einlesen des ersten Wertes, ohne Bedeutung fr die Auswertung, da
    // hier das TOF-Meprogramm nur die Gesamtzahl der Counts abspeichert.
    // --- CHRISTIAN BUTH: Stimmt nicht.  Ist der erste Kanal, da die Gesamtzahl
    // Kanle sonst um eins zu klein ist.  Der Kanal enthlt jedoch nur Mll, so
    // dass er getrost ignoriert werden kann. ---
    fscanf(Dateizeiger,"%f",&wert);
    messdaten[0] = wert;
    fehler[0] = sqrt(wert);

    // Einlesen der relevanten Medaten
    for (i=1; i<=anzahl_kanaele; i++)
      {
	fscanf(Dateizeiger,"%f",&wert);
	messdaten[i] = wert;
	fehler[i] = sqrt(wert);      // Berechnen des statistischen Fehlers
      }

    // Freigeben der benutzten Zeiger
    fclose(Dateizeiger);
    return;
  }


// Aufbereiten des Spektrums: Zuerst wird der Untergrund und sein Fehler
// am linken Rand des Spektrums ermittelt.  Dieser wird dann vom gemeenen
// Spektrum abgezogen.  Als nchstes wird das Spektrum auf 1 normiert.
// Zuletzt werden die Daten auf den Zeit-Null-Punkt (Chopper ganz offen)
// gesetzt.

void Aufbereiten(void)
  {
    float wert, *dummy1, *dummy2;
    int i, oben, unten;

    // Wenn noch keine Medaten geladen wurden.
    if (! anzahl_kanaele)
      {
        printf("Es wurden noch keine Medaten geladen!\n");
        return;
      }

    // Ermitteln des Untergrundes und seines Fehlers anhand des
    // selbst gewhlten Bereiches.
    printf("\nGeben Sie bitte den Bereich des TOF-Spektrums an, aus dem der");
    printf("\nUntergrund ermittelt werden soll: ");
    printf("\nuntere Kanalzahl: ");
    scanf("%d",&unten);
    printf("\nobere Kanalzahl: ");
    scanf("%d",&oben);

    wert = 0;
    for (i=unten; i<=oben; i++)
        wert = wert + messdaten[i];
    untergrund = wert / (oben - unten + 1);
    wert = 0;
    for (i=unten; i<=oben; i++)
        wert = wert + sqr(messdaten[i] - untergrund);
    fehler_untergrund = sqrt(wert / (oben - unten + 1));
    printf("\nDer Untergrund betrug: %f +- %f\n",untergrund,fehler_untergrund);

    // Abziehen des Untergrundes und Fehlerfortpflanzung
    for (i=1; i<=anzahl_kanaele; i++)
      {
   	messdaten[i] = messdaten[i] - untergrund;
        fehler[i] = sqrt(sqr(fehler[i]) + sqr(fehler_untergrund));
      }

    // Ermitteln des Index des Zeit-Null-Punktes in TOF-Spektrum, d.h.
    // des Punktes, an dem der Chopper ganz geffnet war.
    Kanal0 = floor(43.605 / 1401.6 / frequenz / dwell);
    printf("Der Kanal 0 war: %d\n",Kanal0);

    // Allokieren des bentigten Speicherplatzes
    dummy1 = (float *) malloc ((anzahl_kanaele - Kanal0 + 1) * sizeof(float));
    dummy2 = (float *) malloc ((anzahl_kanaele - Kanal0 + 1) * sizeof(float));

    // Verschieben des Spektrums auf diesen Zeit-Null-Punkt
    for (i=1; i<=anzahl_kanaele - Kanal0; i++)
      {
   	dummy1[i] = messdaten[i + Kanal0];
        dummy2[i] = fehler[i + Kanal0];
      }

    free(messdaten);
    free(fehler);

    messdaten = dummy1;
    fehler = dummy2;

    anzahl_kanaele = anzahl_kanaele - Kanal0;

    // Normieren auf Flche 1
    wert = 0;
    for (i=1; i<=anzahl_kanaele; i++)
        wert = wert + messdaten[i];
    for (i=1; i<=anzahl_kanaele; i++)
      {
   	messdaten[i] = messdaten[i] / wert;
        fehler[i] = fehler[i] / wert;
      }
    return;
  }


// Routine zum Abspeichern der verschiedenen erzeugten Spektren auf die
// Festplatte in Abhngigkeit der eingestellten Kanalzahl.  Der gewnschte
// Dateiname wird abgefragt.

void Abspeichern(float *spektrum)
  {
    int i, anzahl = anzahl_kanaele;
    FILE *Dateizeiger;
    char name[Dateinamenlaenge+1], file_name[Pfadnamenlaenge+1], mit_fehler[2],
         anzahl_korrekt[2];

    // Wenn noch keine Medaten geladen wurden.
    if (! anzahl_kanaele)
      {
        printf("Es wurden noch keine Medaten geladen!\n");
        return;
      }

    // Parameter erfragen
    printf("\nGeben Sie bitte den Namen des Files an (maximal "
           make_str_eval(Dateinamenlaenge) " Zeichen): ");
    scanf("%" make_str_eval(Dateinamenlaenge) "s",name);
    printf("\nSind Fehlerbalken mitabzuspeichern (j/n): ");
    scanf("%1s",&mit_fehler);
    printf("\nIst die Anzahl der abzuspeichernden Werte (aktuell: %d) "
           "korrekt (j/n): ",anzahl);
    scanf("%1s",&anzahl_korrekt);
    if (anzahl_korrekt[0] != 'j')
      {
     	printf("\nGeben Sie bitte eine neue Anzahl an: ");
	scanf("%d",&anzahl);
      }

    // Erzeugung des vollstndigen Pfadnamens
    strcpy(file_name, dir);
    strcat(file_name, name);
    strcat(file_name, Ausgabe_Endung);

    // ffnen der Datei und berprfung, ob existent
    Dateizeiger=fopen(file_name,"wt");

    // Abspeichern des Spektrums
    if (mit_fehler[0] != 'j')
      {
     	for (i=1; i<=anzahl; i++)
	    fprintf(Dateizeiger,"%d   %.8E\n", i, spektrum[i]);
      }
    else
      {
     	for (i=1; i<=anzahl; i++)
	    fprintf(Dateizeiger,"%d   %.8E   %.8E\n", i,spektrum[i],fehler[i]);
      }

    // Freigeben der benutzten Zeiger
    fclose(Dateizeiger);
    return;
  }


// Routine zum Auflisten von Daten auf dem Bildschirm
// in Abhngigkeit der eingestellten Kanalzahl.

void Ausgabe(float *tof)
  {
    int i;

    // Wenn noch keine Medaten geladen wurden.
    if (! anzahl_kanaele)
      {
        printf("Es wurden noch keine Medaten geladen!\n");
        return;
      }

    // Medaten ausgeben
    for (i=1; i<=anzahl_kanaele; i++)
        printf("Kanal %i = %.3E \n", i, tof[i]);
    return;
  }


// Erzeugung des theoretischen Flugzeitspektrums in Abhngigkeit der
// eingestellten Temperatur [K] und Flugstrecke [m].  Das Spektrum wird in
// Abhngigkeit von der Kanalzahl (i*dwell) der TOF-Meung erzeugt, um
// einen einfachen Fit und einen direkten Vergleich mit den gemeenen
// Daten zu ermglichen.

// Das Spektrum wird mittels Gleichung (4.4) des Skriptes berechnet.  Das
// Spektrum wird auf 1 normiert.

void Theoretisches_Spektrum(float T,int anzahl)
  {
    register int i;
    double t,integral=0;

    // Theoretisches Spektrum und Integral darber berechnen
    for (i=1; i<=anzahl; i++)
      {
        t = dwell * i;
        spektrum_theo[i] = pow(t,-5.0) * exp(-m * sqr(flugstrecke)
                           / (2.0 * k * T * sqr(t)));
        integral += spektrum_theo[i];
      }
    // Spektrum normieren
    for (i=1; i<=anzahl; i++)
        spektrum_theo[i] /= integral;
    return;
  }


// Berechnen des aktuellen Chi-Quadrates

float chi_quadrat_berechnen()
  {
    register int i;
    double chi_sq=0,delta;

    for (i=1; i<=anzahl_daten; i++)
      {
        delta = (messdaten[i] - spektrum_theo[i]) / fehler[i];
        chi_sq += sqr(delta);
      }
    return(chi_sq);
  }


// Routine zum mehrfachen Berechnen des Chi-Quadrates bei Variation der
// Temperatur

void chi_quadrat_Parabel()
  {
    register int i;
    float T;

    // Wenn noch keine Medaten geladen wurden.
    if (! anzahl_kanaele)
      {
        printf("Es wurden noch keine Medaten geladen!\n");
        return;
      }

    // Parameter einlesen
    printf("Ab welchem Kanal wird nur noch Untergrund beobachtet "
           "(Kanal0 = %d): ", Kanal0);
    scanf("%d",&anzahl_daten);
    printf("Bei welcher Temperatur soll die chi^2 Parabel beginnen: ");
    scanf("%f", &T_start);
    printf("Welche Temperaturschrittweite wird gewnscht: ");
    scanf("%f", &Delta_T);
    printf("Wieviele Temperaturschritte sollen erfolgen: ");
    scanf("%d", &anzahl_schritte);

    // Speicherplatz fr die Berechnungen belegen
    free(chi_quadrat);
    spektrum_theo = (float *) malloc((anzahl_daten+1) * sizeof(float));
    chi_quadrat = (float *) malloc((anzahl_schritte+1) * sizeof(float));

    /* chi^2 Parabel berechnen */
    for (i=1; i<=anzahl_schritte; i++)
      {
        T = (i-1) * Delta_T + T_start;
        Theoretisches_Spektrum(T,anzahl_daten);
        chi_quadrat[i] = chi_quadrat_berechnen();
      }

    free(spektrum_theo);
    return;
  }


// Routine zum Abspeichern der erzeugten ch^2 Parabeln auf die Festplatte in
// Abhngigkeit des eingestellten Temperaturbereichs.  Der gewnschte
// Dateiname wird abgefragt.

void Abspeichern_der_chi_quadrat_Parabel()
  {
    int i;
    FILE *Dateizeiger;
    char name[Dateinamenlaenge+1], file_name[Pfadnamenlaenge+1];
    float T;

    // Wenn noch kein Fit gemacht wurde zurck.
    if (! chi_quadrat)
      {
        printf("Es wurde noch kein Fit generiert!\n");
        return;
      }

    printf("\nGeben Sie bitte den Namen des Files an (maximal "
           make_str_eval(Dateinamenlaenge) " Zeichen): ");
    scanf("%" make_str_eval(Dateinamenlaenge) "s",name);

    // Erzeugung des vollstndigen Pfadnamens
    strcpy(file_name, dir);
    strcat(file_name, name);
    strcat(file_name, Ausgabe_Endung);

    // ffnen der Datei und berprfung, ob existent
    Dateizeiger=fopen(file_name,"wt");

    // Abspeichern der Parabel
    for (i=1; i<=anzahl_schritte; i++)
      {
        T = (i-1) * Delta_T + T_start;
	fprintf(Dateizeiger,"%.8E   %.8E\n", T, chi_quadrat[i]);
      }

    // Freigeben des benutzten Zeigers
    fclose(Dateizeiger);
    return;
  }


// Routine zum Abspeichern eines theoretischen Spektrums fr eine feste
// Temperatur auf die Festplatte.  Der gewnschte Dateiname wird abgefragt.

void Abspeichern_des_theoretischen_Spektrums()
  {
    int i,anzahl;
    FILE *Dateizeiger;
    char name[Dateinamenlaenge+1], file_name[Pfadnamenlaenge+1];
    float T;

    // Daten erfragen
    printf("Fr welche Temperatur soll das Spektrum erzeugt werden: ");
    scanf("%f",&T);
    printf("Fr wieviele Kanle sollen Werte generiert werden: ");
    scanf("%d",&anzahl);

    printf("\nGeben Sie bitte den Namen des Files an (maximal "
           make_str_eval(Dateinamenlaenge) " Zeichen): ");
    scanf("%" make_str_eval(Dateinamenlaenge) "s",name);

    // Erzeugung des vollstndigen Pfadnamens
    strcpy(file_name, dir);
    strcat(file_name, name);
    strcat(file_name, Ausgabe_Endung);

    // ffnen der Datei und berprfung, ob existent
    Dateizeiger = fopen(file_name,"wt");

    // Theoretisches Spektrum generieren
    spektrum_theo = (float *) malloc((anzahl+1) * sizeof(float));
    Theoretisches_Spektrum(T,anzahl);

    // Abspeichern des theoretischen Spektrums
    for (i=1; i<=anzahl; i++)
	fprintf(Dateizeiger,"%d   %.8E\n", i, spektrum_theo[i]);

    // Freigeben der benutzten Zeiger
    fclose(Dateizeiger);
    free(spektrum_theo);
    return;
  }


// Untermen zum ndern der eingestellten Parameter

void Parameter(void)
  {
    int a=1;

    while (a != 0)
      {
        printf("\n");
	printf("ndern der Parameter (Ende mit 0):\n\n");

        printf("Aktuelle Parameter: Anzahl der Kanle: %i\n", anzahl_kanaele);
        printf("                    Directory:         %s\n", dir);
        printf("                    Flugstrecke:       %1.3f m\n", flugstrecke);
        printf("                    Chopperfrequenz:   %3.2f Hz\n", frequenz);
        printf("                    dwelltime:         %2.2e s\n", dwell);
	printf("                    Temperatur:        %3.1f K\n\n",temperatur);

	printf("0: Zurck\n");
	printf("1: Anzahl der Kanle\n");
        printf("2: Flugstrecke\n");
        printf("3: Chopperfrequenz\n");
        printf("4: dwelltime\n");
        printf("5: Temperatur\n");
        printf("6: Directory ndern\n");

	printf("\nWelchen Parameter mchten Sie ndern: ");
	scanf("%d",&a);
	printf("\n");

	switch (a)
	  {
     	    case 0:
                break;
	    case 1:
            	printf("\nNeue Anzahl der Kanle: ");
		scanf("%d",&anzahl_kanaele);
		break;
	    case 2:
            	printf("\nNeue Flugstrecke: ");
		scanf("%f",&flugstrecke);
		break;
	    case 3:
            	printf("\nNeue Chopperfrequenz: ");
		scanf("%f",&frequenz);
		break;
	    case 4:
            	printf("\nNeue dwelltime: ");
		scanf("%f",&dwell);
		break;
	    case 5:
            	printf("\nNeue Temperatur: ");
		scanf("%f",&temperatur);
		break;
     	    case 6:
            	printf("\nNeues Directory: ");
	    	scanf("%" make_str_eval(Pfadlaenge) "s",&dir);
		break;
	    default:
            	printf("Falsche Eingabe!!!\n");
                break;
          }
      }
    printf("\n");
    return;
  }


// Hauptprogramm, das ein Men anbietet, um die verschiedenen Unterprogramme
// aufrufen zu knnen.

int main()
  {
    int a=1;

    while (a != 0)
      {
        printf("\n");
	printf("Programm: Entfaltung der TOF-Spektren\n"
               "(c) Christian Buth und Martin Klein\n\n");

        printf("aktuelle Parameter: Anzahl der Kanle: %i\n", anzahl_kanaele);
        printf("                    Directory:          %s\n", dir);
        printf("                    Flugstrecke:        %1.3f m\n",flugstrecke);
        printf("                    Chopperfrequenz:    %3.2f Hz\n", frequenz);
        printf("                    dwelltime:          %2.2e s\n", dwell);
        printf("                    Temperatur:         %3.1f K\n\n",
                                                                    temperatur);

        printf("0: Programm beenden\n");
	printf("1: Laden eines Spektrums\n");
   	printf("2: Aufbereiten des gemeenen Spektrums\n");
        printf("3: Abspeichern des aufbereiteten/gemeenen Spektrums\n");
        printf("4: Anzeigen des aufbereiteten/gemeenen Spektrums\n");
        printf("5: ndern der Parameter\n");

        printf("6: chi^2 ber einen Temperaturbereich berechnen\n");
        printf("7: chi^2 Parabel abspeichern\n");
        printf("8: Theoretisches Spektrum erzeugen\n");

	printf("\nWas mchten Sie: ");
	scanf("%d",&a);
	printf("\n");

	switch (a)
	  {
     	    case 0:
            	break;
	    case 1:
            	Laden();
		break;
            case 2:
            	Aufbereiten();
	     	break;
     	    case 3:
            	Abspeichern(messdaten);
	     	break;
            case 4:
            	Ausgabe(messdaten);
	    	break;
     	    case 5:
            	Parameter();
	    	break;
            case 6:
                chi_quadrat_Parabel();
                break;
            case 7:
                Abspeichern_der_chi_quadrat_Parabel();
                break;
            case 8:
                Abspeichern_des_theoretischen_Spektrums();
                break;
	    default:
            	printf("Falsche Eingabe!!!\n");
                break;
          }
      }

    printf("\n");
    free(messdaten);
    free(fehler);
    free(chi_quadrat);
    return(EXIT_SUCCESS);
}
